
/**
 * Controls the amount of console log output generated by the Amazon GameLift Streams WebSDK
 * @param level - the desired console logging level ("none" or "debug")
 */
export declare function setLogLevel(level: "none" | "debug"): void;
/**
 * Performance stats for the application running in the current session
 * For normalized metrics, 0.0 - 1.0 indicates usage within fair share (ideal).
 * Values above 1.0 may cause performance issues on multi-tenant stream classes.
 */
export type ApplicationStats = {
	/** Normalized application CPU utilization */
	cpuUtilizationNormalized?: number;
	/** Normalized application memory utilization */
	memoryUtilizationNormalized?: number;
	/** Application memory utilization in mebibytes (MiB) */
	memoryUtilizationMb?: number;
	[key: string]: number | undefined;
};
/**
 * Performance stats for the shared compute that is serving the current session
 * On multi-tenant stream groups, this compute may be shared among multiple sessions
 */
export type SharedSystemStats = {
	/** System CPU utilization as percentage (0-100) */
	cpuUtilization?: number;
	/** System Memory utilization as percentage (0-100) */
	memoryUtilization?: number;
	/** System Memory utilization in mebibytes (MiB) */
	memoryUtilizationMb?: number;
	/** System GPU utilization as percentage (0-100) */
	gpuUtilization?: number;
	/** System VRAM utilization as percentage (0-100) */
	vramUtilization?: number;
	/** System VRAM utilization in mebibytes (MiB) */
	vramUtilizationMb?: number;
	[key: string]: number | undefined;
};
/**
 * Provides both application-level and system-level performance stats
 */
export interface PerformanceStats {
	/** Timestamp representing when the stats were captured */
	timestamp: Date;
	/** Performance stats for the application running in the current session */
	application: ApplicationStats;
	/** Performance stats for the shared compute that is serving the current session */
	system: SharedSystemStats;
}
export interface IClientConnection {
	/**
	 * The connectionState callback is triggered whenever the RTCPeerConnection connectionState value changes.
	 * @see {@link https://developer.mozilla.org/en-US/docs/Web/API/RTCPeerConnection}
	 * @param state - "connecting" | "connected" | "disconnected" | "failed" | "closed"
	 */
	connectionState?: (state: string) => void;
	/**
	 * The channelError callback is triggered whenever there is an internal WebRTC data channel error.
	 * Any such errors are usually unrecoverable and your code should terminate the streaming session.
	 * @param e - error object reported by browser WebRTC implementation
	 */
	channelError?: (e: any) => void;
	/**
	 * The serverDisconnect callback is triggered when the streaming server initiates a disconnection.
	 * @param reasoncode - "idle" | "terminated" | (other strings reserved for future use)
	 */
	serverDisconnect?: (reasoncode: string) => void;
	/**
	 * The applicationMessage callback is triggered whenever a message is received from the application through the data channel.
	 * @param message - message received from application (as byte array)
	 */
	applicationMessage?: (message: Uint8Array) => void;
	/**
	 * The performanceStats callback is triggered whenever the latest performance stats for the current session are received
	 * Stream Session must be configured to share performance stats
	 * @param PerformanceStats - Snapshot of application-level and system-level performance stats
	 */
	performanceStats?: (stats: PerformanceStats) => void;
}
export interface GamepadFilterResult {
	/**
	 * An array of button indices that should be suppressed when forwarding gamepad input
	 * to the stream. Each array element should be in the [0, 16] range according to the
	 * Standard gamepad mapping, as defined in the W3C Gamepad specification.
	 * @see {@link https://w3c.github.io/gamepad/#remapping}
	 */
	suppressButtonIndexes: ReadonlyArray<number>;
}
export interface InputConfiguration {
	/**
	 * Automatically listen for and forward keyboard events when input is attached
	 * @defaultValue true
	 */
	autoKeyboard?: boolean;
	/**
	 * Automatically listen for and forward mouse events
	 * @defaultValue true
	 */
	autoMouse?: boolean;
	/**
	 * Automatically look for connected gamepads and forward gamepad input every frame
	 * @defaultValue true
	 */
	autoGamepad?: boolean;
	/**
	 * Reset all input device state whenever detachInput is called. If false, keyboard and mouse
	 * buttons will be frozen in their current state (down vs up) when detachInput() is called.
	 * @defaultValue true
	 */
	resetOnDetach?: boolean;
	/**
	 * Detaches input devices automatically when the window loses focus, and restores input
	 * devices when the window regains focus. If false, Amazon GameLift Streams ignores window focus and you
	 * must call detachInput() manually to stop tracking user input devices and attachInput()
	 * to resume. This setting combines with resetOnDetach to automatically release any held
	 * keys or buttons when window focus is lost.
	 * @defaultValue true
	 */
	trackWindowFocus?: boolean;
	/**
	 * If set, the keyboardFilter callback function is used to filter every keyboard event.
	 * The callback should return 'false' to prevent the event from being forwarded to the stream.
	 * @defaultValue null
	 */
	keyboardFilter?: null | ((event: KeyboardEvent) => boolean);
	/**
	 * If set, the mouseFilter callback function is used to filter every mouse event.
	 * The callback should return 'false' to prevent the event from being forwarded to the stream.
	 * @defaultValue null
	 */
	mouseFilter?: null | ((event: MouseEvent) => boolean);
	/**
	 * If set, the gamepadFilter callback function is used to filter gamepad inputs.
	 * The callback should return 'false' to completely suppress the gamepad, or a
	 * GamepadFilterResult object to suppress only individual buttons.
	 * @defaultValue null
	 */
	gamepadFilter?: null | ((event: Gamepad) => boolean | GamepadFilterResult);
	/**
	 * Allow haptic feedback events (eg, controller vibration) to be triggered by the stream
	 * @defaultValue true
	 */
	hapticFeedback?: boolean;
	/**
	 * Allow mouse cursor shape to be controlled by the stream
	 * Options:
	 *   true - the local cursor replicates the remote cursor shape on the stream host
	 *   false - the local cursor is never modified or hidden
	 *   'visibility' - the local cursor is never modified, but it can be hidden
	 *   function - the local cursor is never modified, but the function is invoked with a CSS style string representing the remote cursor whenever the remote cursor changes
	 * @defaultValue 'visibility'
	 */
	setCursor?: true | false | "visibility" | ((cursorStyle: string) => void);
	/**
	 * Automatically capture the mouse (hiding the cursor and locking the mouse to the current window)
	 * whenever the remote cursor has been made invisible on the stream host
	 * Options:
	 *   true - the mouse is always captured
	 *   false - the mouse is never captured
	 *   'fullscreen' - the mouse is only captured if the video element is fullscreen
	 *
	 * This is often helpful for "first-person" games which use the mouse for aiming.
	 *
	 * This has no effect in the Safari browser or on the iOS platform due to platform limitations.
	 * @defaultValue 'fullscreen'
	 */
	autoPointerLock?: true | false | "fullscreen";
}
/**
 * Amazon GameLift Streams constructor configuration object
 */
export interface GameLiftStreamsArgs {
	/** HTML video element which will be used to display the stream */
	videoElement: HTMLVideoElement;
	/** HTML audio element which will be used to play the stream audio*/
	audioElement: HTMLAudioElement;
	/** Mouse/keyboard/gamepad configuration properties */
	inputConfiguration?: InputConfiguration;
	/** Connection callbacks */
	clientConnection?: IClientConnection;
	/** Stream configuration properties */
	streamConfiguration?: GameLiftStreamsStreamConfiguration;
}
/**
 * Amazon GameLift Streams stream configuration properties
 */
export interface GameLiftStreamsStreamConfiguration {
	/** Enable audio stream (default: true) */
	enableAudio?: boolean;
	/** Maximum client download bitrate, as kilobits-per-second (default: undefined) */
	maximumKbps?: number;
	/** Maximum client-supported video resolution, regardless of game resolution (default: 1080p) */
	maximumResolution?: "1080p" | "720p" | "540p";
	/**
	 * [EXPERIMENTAL] Client-supplied ICE server information to include in WebRTC connection negotiation. This
	 * setting is experimental and may be removed in the future. It is not recommended outside of specific
	 * enterprise scenarios where clients are known to be behind a restrictive firewall. (default: undefined)
	 */
	iceServers?: GameLiftStreamsIceServerConfiguration[] | (() => GameLiftStreamsIceServerConfiguration[]) | (() => Promise<GameLiftStreamsIceServerConfiguration[]>);
	/**
	 * [EXPERIMENTAL] Force negotiation of a specific WebRTC video codec configuration. This setting is
	 * experimental and may be removed in the future. It should only be set as part of a customer support
	 * investigation into client compatibility with new hardware platforms. (default: undefined)
	 */
	forceVideoCodec?: "h264.constrained.31" | "h264.constrained" | "h264.baseline" | "h264.main.31" | "h264.main" | "h264.high";
}
/** Amazon GameLift Streams STUN/TURN server configuration, allowing either 'urls' or 'url' */
export type GameLiftStreamsIceServerConfiguration = RTCIceServer | {
	/** URL representing this server, starting with 'turn:' or 'turns:' or 'stun:' or 'stuns:' */
	url: string;
	/** Username for TURN server authentication */
	username?: string;
	/** Credential (password) for TURN server authentication */
	credential?: string;
};
/**
 * Manages the connection between an existing HTML video element and a single Amazon GameLift Streams stream
 */
export declare class GameLiftStreams {
	#private;
	/**
	 * @param args - GameLiftStreamsArgs configuration object
	 */
	constructor(args: GameLiftStreamsArgs);
	/**
	 * Unique 128-bit identifier (as a 16-character hex string)
	 */
	get id(): string;
	/**
	 * Attempts to attach microphone input to the next Amazon GameLift Streams session. This may trigger UI
	 * in the web browser which asks the user if they wish to allow microphone access.
	 * Note that this operation must be completed/resolved before generateSignalRequest().
	 * @param constraints - optional device selection criteria, such as deviceId (see https://developer.mozilla.org/en-US/docs/Web/API/MediaTrackConstraints)
	 * @returns Promise<void> that resolves when the web browser has allowed microphone access
	 * @throws DOMException if the web browser denies microphone access
	 */
	enableMicrophone(constraints?: MediaTrackConstraints): Promise<void>;
	/**
	 * Prepares to connect to a Amazon GameLift Streams session by internally generating a WebRTC offer and
	 * converting that offer into a SignalRequest string value. The resulting string should
	 * be passed to your backend server, which will forward the string as the SignalRequest
	 * parameter to a Amazon GameLift Streams StartStreamSession API call.
	 * @see {@link https://developer.mozilla.org/en-US/docs/Web/API/WebRTC_API/Connectivity}
	 * @returns Promise<string> that resolves into a SignalRequest string value
	 * @throws Error if browser is incompatible due to lack of WebRTC or video codec support
	 */
	generateSignalRequest(): Promise<string>;
	/**
	 * Completes the connection to a Amazon GameLift Streams stream by parsing the SignalResponse string
	 * retrieved by your backend server, internally transforming it into a WebRTC answer,
	 * and then completing the WebRTC stream connection from browser to Amazon GameLift Streams.
	 * @see {@link https://developer.mozilla.org/en-US/docs/Web/API/WebRTC_API/Connectivity}
	 * @param signalResponse - SignalResponse value retrieved by your backend server
	 * @returns Promise<void> that resolves when the stream has finished connecting
	 */
	processSignalResponse(signalResponse: string): Promise<void>;
	/**
	 * Enables mouse/keyboard/gamepad input event transmission and enables any
	 * automatic event listeners that were specified in the input configuration
	 * passed to the Amazon GameLift Streams constructor.
	 * Only valid after the stream connection is established and before calling close().
	 */
	attachInput(): void;
	/**
	 * Disables mouse/keyboard/gamepad input event transmission and disables any
	 * automatic input listeners that were specified in the input configuration
	 * passed to the Amazon GameLift Streams constructor.
	 * Only valid after the stream connection is established and before calling close().
	 */
	detachInput(): void;
	/**
	 * Retrieves the WebRTC RTCStatsReport objects with browser-specific stream metrics
	 * for the video track.
	 * @returns Promise<RTCStatsReport> which resolves to a WebRTC RTCStatsReport object
	 */
	getVideoRTCStats(): Promise<RTCStatsReport>;
	/**
	 * Retrieves the WebRTC RTCStatsReport objects with browser-specific stream metrics
	 * for the audio track.
	 * @returns Promise<RTCStatsReport> which resolves to a WebRTC RTCStatsReport object
	 */
	getAudioRTCStats(): Promise<RTCStatsReport>;
	/**
	 * Retrieves the WebRTC RTCStatsReport objects with browser-specific stream metrics
	 * for the microphone audio track.
	 * @returns Promise<RTCStatsReport> which resolves to a WebRTC RTCStatsReport object
	 */
	getMicrophoneRTCStats(): Promise<RTCStatsReport>;
	/**
	 * Retrieves the WebRTC RTCStatsReport objects with browser-specific stream metrics
	 * for the input data channel.
	 * @returns Promise<RTCStatsReport> which resolves to a WebRTC RTCStatsReport object
	 */
	getInputRTCStats(): Promise<RTCStatsReport>;
	/**
	 * Closes the stream and destroys internal state of this Amazon GameLift Streams object.
	 */
	close(): void;
	/**
	 * Transmits a KeyboardEvent (keydown or keyup) to the remote side of the stream.
	 * This is not necessary if the autoKeyboard input configuration property was set.
	 * @param e - the event to pass
	 * @returns true if the event was transmitted to the remote side of the stream
	 */
	processKeyboardEvent(e: KeyboardEvent): boolean;
	/**
	 * Transmits a MouseEvent (mousedown, mouseup, mousemove, or mousewheel) to the remote side of the stream.
	 * This is not necessary if the autoMouse input configuration property was set.
	 * @param e - the event to pass
	 * @returns true if the event was transmitted to the remote side of the stream
	 */
	processMouseEvent(e: MouseEvent): boolean;
	/**
	 * Adds a gamepad to the set of tracked gamepads. This can be a gamepad from
	 * the browser's navigator.getGamepads() API, or a fake gamepad-like object.
	 * This is not necessary if the autoGamepad input configuration property was set.
	 *
	 * If adding a gamepad-like object, you should set the .index property of the
	 * object to be a high number (100+) to avoid conflicting with the .index
	 * property of any real gamepads.
	 *
	 * @see {@link https://developer.mozilla.org/en-US/docs/Web/API/Gamepad}
	 * @param gamepad - the W3C Gamepad or gamepad-like object to add
	 * @returns true if the gamepad was added to the set of tracked gamepads
	 */
	addGamepad(gamepad: Gamepad): boolean;
	/**
	 * Remove a gamepad from the set of tracked gamepads.
	 * This is not necessary if the autoGamepad input configuration property was set.
	 * @param gamepad - the W3C Gamepad or gamepad-like object to remove
	 * @returns true if the gamepad was removed from the set of tracked gamepads
	 */
	removeGamepad(gamepad: Gamepad): boolean;
	/**
	 * Scan all tracked gamepads for any input changes forward the updated inputs to
	 * the remote side of the stream.
	 * This is not necessary if the autoGamepad input configuration property was set.
	 * @returns true if the stream is connected and there were no internal errors
	 */
	processGamepads(): boolean;
	/**
	 * Send a message (as byte array) to the game application.
	 * @param message - the message to send as byte array
	 * @returns true if the message was sent to the application
	 */
	sendApplicationMessage(message: Uint8Array): boolean;
}
export declare const VERSION = "1.1.0";

export as namespace gameliftstreams;

export {};
